package gov.va.genisis2.ts.service.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.jena.query.QueryExecution;
import org.apache.jena.query.QueryExecutionFactory;
import org.apache.jena.query.QuerySolution;
import org.apache.jena.query.ResultSet;
import org.apache.jena.rdf.model.Literal;
import org.apache.jena.rdf.model.RDFNode;
import org.apache.jena.rdf.model.Resource;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import gov.va.genisis2.ts.common.dto.DataElementDTO;
import gov.va.genisis2.ts.fuseki.FusekiClient;
import gov.va.genisis2.ts.service.IDataElementService;
import gov.va.genisis2.ts.utils.TSPropertiesUtil;

@Service
public class DataElementService implements IDataElementService {
	
	private static final Logger LOGGER = LogManager.getLogger(DataElementService.class);
	
	@Autowired
	private FusekiClient fusekiClient;
	
	@Autowired
	private TSPropertiesUtil propsUtil;
	
	private final String URI_HEAD_VAR = "uri";
	private final String NAME_HEAD_VAR = "name";

	@Override
	public List<DataElementDTO> getDataElements() {
		
		QueryExecution qe = null;
		ResultSet resultSet = null;
		List<DataElementDTO> dataElements = new ArrayList<DataElementDTO>();
		
		try {
			String dataElementSparqlQuery = propsUtil.getDataElementSparqlQuery();
			qe = QueryExecutionFactory.sparqlService(propsUtil.getSparqlEndpoint(), dataElementSparqlQuery);
			resultSet = fusekiClient.performSelectQuery(qe, propsUtil.getSparqlEndpoint());

			if (null != resultSet) {
				while (resultSet.hasNext()) {

					QuerySolution binding = resultSet.nextSolution();
					DataElementDTO dataElement = new DataElementDTO();
					
					//should be a URI
					if( binding.contains( URI_HEAD_VAR )) {
						RDFNode uriNode = binding.get( URI_HEAD_VAR );
						if( uriNode != null && uriNode.isURIResource() ){
							Resource uriNodeResource =  uriNode.asResource();
							if( uriNodeResource != null ) {
								dataElement.setUri(uriNodeResource.getURI().trim());
							}
						}
					}
					//should be a literal
					if( binding.contains( NAME_HEAD_VAR ) ) {
						RDFNode literalNode = binding.get( NAME_HEAD_VAR );
						if( literalNode != null && literalNode.isLiteral()) {
							Literal literal = literalNode.asLiteral();
							if( literal != null ) {
								dataElement.setLabel(literal.toString().trim());
							}
						}
					}
					dataElements.add(dataElement);
				}
			}
			
			//Sort the elements based on the label name
			Collections.sort(dataElements, (a, b) -> a.getLabel().compareTo(b.getLabel()));
			
			return dataElements;
		} catch (Exception e) {
			LOGGER.error("Error in execution Data Element SPARQL query", e);
			throw e;
		} finally {
			if (qe != null && !qe.isClosed()) {
				qe.close();
			}
		}
	}

}
